/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.net.ssl;

import java.io.*;
import java.net.*;
import java.nio.channels.*;
import javax.net.ssl.*;

/**
 * Wrapper for sockets which enables to add functionality in subclasses on top
 * of existing, connected SSL sockets. It is useful when direct subclassing of
 * delegate SSL socket class is not possible, e.g. if the delegate socket is
 * created by a library. Possible usage example is socket factory chaining. This
 * class delegates all socket-related requests to the wrapped delegate, as of
 * JDK 1.4.
 *
 * @author Dawid Kurzyniec
 * @version 1.4
 */
public class SSLSocketWrapper extends SSLSocket {

    /**
     * the wrapped delegate socket.
     */
    protected final SSLSocket delegate;

    /**
     * Creates new socket wrapper for a given SSL socket. The delegate
     * must be connected and bound and it must not be closed.
     * @param delegate the delegate SSL socket to wrap.
     * @throws SocketException if the delegate socket is closed, not bound,
     *                         or not connected.
     */
    protected SSLSocketWrapper(SSLSocket delegate) throws SocketException {
        if (delegate == null) {
            throw new NullPointerException();
        }
        if (delegate.isClosed()) {
            throw new SocketException("Delegate server socket is closed");
        }
        if (!(delegate.isBound())) {
            throw new SocketException("Delegate server socket is not bound");
        }
        if (!(delegate.isConnected())) {
            throw new SocketException("Delegate server socket is not connected");
        }
        this.delegate = delegate;
    }

    public void connect(SocketAddress endpoint) throws IOException {
        delegate.connect(endpoint);
    }

    public void connect(SocketAddress endpoint, int timeout) throws IOException {
        delegate.connect(endpoint, timeout);
    }

    public void bind(SocketAddress bindpoint) throws IOException {
        delegate.bind(bindpoint);
    }

    public InetAddress getInetAddress() {
        return delegate.getInetAddress();
    }

    public InetAddress getLocalAddress() {
        return delegate.getLocalAddress();
    }

    public int getPort() {
        return delegate.getPort();
    }

    public int getLocalPort() {
        return delegate.getLocalPort();
    }

    public SocketAddress getRemoteSocketAddress() {
        return delegate.getRemoteSocketAddress();
    }

    public SocketAddress getLocalSocketAddress() {
        return delegate.getLocalSocketAddress();
    }

    public SocketChannel getChannel() {
        return delegate.getChannel();
    }

    public InputStream getInputStream() throws IOException {
        return delegate.getInputStream();
    }

    public OutputStream getOutputStream() throws IOException {
        return delegate.getOutputStream();
    }

    public void setTcpNoDelay(boolean on) throws SocketException {
        delegate.setTcpNoDelay(on);
    }

    public boolean getTcpNoDelay() throws SocketException {
        return delegate.getTcpNoDelay();
    }

    public void setSoLinger(boolean on, int linger) throws SocketException {
        delegate.setSoLinger(on, linger);
    }

    public int getSoLinger() throws SocketException {
        return delegate.getSoLinger();
    }

    public void sendUrgentData(int data) throws IOException  {
        delegate.sendUrgentData(data);
    }

    public void setOOBInline(boolean on) throws SocketException {
        delegate.setOOBInline(on);
    }

    public boolean getOOBInline() throws SocketException {
        return delegate.getOOBInline();
    }

    public void setSoTimeout(int timeout) throws SocketException {
        delegate.setSoTimeout(timeout);
    }

    public int getSoTimeout() throws SocketException {
        return delegate.getSoTimeout();
    }

    public void setSendBufferSize(int size) throws SocketException{
        delegate.setSendBufferSize(size);
    }

    public int getSendBufferSize() throws SocketException {
        return delegate.getSendBufferSize();
    }

    public void setReceiveBufferSize(int size) throws SocketException{
        delegate.setReceiveBufferSize(size);
    }

    public int getReceiveBufferSize() throws SocketException {
        return delegate.getReceiveBufferSize();
    }

    public void setKeepAlive(boolean on) throws SocketException {
        delegate.setKeepAlive(on);
    }

    public boolean getKeepAlive() throws SocketException {
        return delegate.getKeepAlive();
    }

    public void setTrafficClass(int tc) throws SocketException {
        delegate.setTrafficClass(tc);
    }

    public int getTrafficClass() throws SocketException {
        return delegate.getTrafficClass();
    }

    public void setReuseAddress(boolean on) throws SocketException {
        delegate.setReuseAddress(on);
    }

    public boolean getReuseAddress() throws SocketException {
        return delegate.getReuseAddress();
    }

    public void close() throws IOException {
        delegate.close();
    }

    public void shutdownInput() throws IOException {
        delegate.shutdownInput();
    }

    public void shutdownOutput() throws IOException {
        delegate.shutdownOutput();
    }

    /**
     * Returns true, indicating that the socket is connected.
     *
     * @return true
     */
    public boolean isConnected() {
        return true;
    }

    /**
     * Returns true, indicating that the socket is bound.
     *
     * @return true
     */
    public boolean isBound() {
        return true;
    }

    public boolean isClosed() {
        return delegate.isClosed();
    }

    public boolean isInputShutdown() {
        return delegate.isInputShutdown();
    }

    public boolean isOutputShutdown() {
        return delegate.isOutputShutdown();
    }

    // SSL-specific methods

    public String[] getSupportedCipherSuites() {
        return delegate.getSupportedCipherSuites();
    }

    public String[] getEnabledCipherSuites() {
        return delegate.getEnabledCipherSuites();
    }

    public void setEnabledCipherSuites(String[] suites) {
        delegate.setEnabledCipherSuites(suites);
    }

    public String[] getSupportedProtocols() {
        return delegate.getSupportedProtocols();
    }

    public String[] getEnabledProtocols() {
        return delegate.getEnabledProtocols();
    }

    public void setEnabledProtocols(String[] protocols) {
        delegate.setEnabledProtocols(protocols);
    }

    public SSLSession getSession() {
        return delegate.getSession();
    }

    public void addHandshakeCompletedListener(
        HandshakeCompletedListener handshakeCompletedListener)
    {
        delegate.addHandshakeCompletedListener(handshakeCompletedListener);
    }

    public void removeHandshakeCompletedListener(
        HandshakeCompletedListener handshakeCompletedListener)
    {
        delegate.removeHandshakeCompletedListener(handshakeCompletedListener);
    }

    public void startHandshake() throws IOException {
        delegate.startHandshake();
    }

    public void setUseClientMode(boolean mode) {
        delegate.setUseClientMode(mode);
    }

    public boolean getUseClientMode() {
        return delegate.getUseClientMode();
    }

    public void setNeedClientAuth(boolean need) {
        delegate.setNeedClientAuth(need);
    }

    public boolean getNeedClientAuth() {
        return delegate.getNeedClientAuth();
    }

    public void setWantClientAuth(boolean want) {
        delegate.setWantClientAuth(want);
    }

    public boolean getWantClientAuth() {
        return delegate.getWantClientAuth();
    }

    public void setEnableSessionCreation(boolean enable) {
        delegate.setEnableSessionCreation(enable);
    }

    public boolean getEnableSessionCreation() {
        return delegate.getEnableSessionCreation();
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof SSLSocketWrapper)) return false;
        SSLSocketWrapper that = (SSLSocketWrapper)obj;
        return this.delegate.equals(that.delegate);
    }

    public int hashCode() {
        return delegate.hashCode() ^ 0x01010101;
    }
}
